﻿/*
VERSION:		1.9
	
USAGE:
	#include "map.as"
	my_map = makeMap( "map_mc", this, 42 );		// creates a 20x15 map at depth 42
	
FUNCTIONS:
	makeMap()		Creates the map system.
	load()				Loads, converts, resets, and draws a map, based on an XML file.  It also loads the collision data.
	drawObj()			Resets & draws a map, based on the data provided.  It also imports the collision data.
	setTile()			Draws a tile.
	reset()				Clears all internal data, and deletes all layers.
	loadChipset()		Loads a chipset.
	
EVENTS:
	onLoad()			Runs after a chipset is loaded.
	
VARIABLES:
	width						Width of the map, in tiles.
	height					Height of the map, in tiles.
	collision_array		2D array containing collision values.
	chipset_pic				Bitmap containing chipset palette.
	layer0_pic				Image containing all tiles on layer 0.		(maps can have any amount of layers)
	layer0_mc				Movieclip displaying layer0_pic.
	layer1_pic				Image containing all tiles on layer 1.
	layer1_mc				Movieclip displaying layer1_pic.
	
NOTE:
	XML data should be saved in a <map> tag, like so:
	writeXml( myData, myXml, "map" );
	
NOTE:
	Layers are assigned depths in incriments of 10
		0 = 0
		1 = 10
		2 = 20
	This allows movieClips to be inserted between them.
	
*/
makeMap = function( instanceName, target_mc, depth, width, height )
{
	#include "nextDepth.as"
	
	// resolve optional parameters
	var target_mc = (target_mc) ? target_mc : this;
	var depth = (depth != undefined) ? depth : nextDepth(target_mc);
	var width = (width) ? width : 20;
	var height = (height) ? height : 15;
	
	// create container movieClip
	var _this = target_mc.createEmptyMovieClip( instanceName, depth );
	
	// define internal variables
	_this.width = width;
	_this.height = height;
	
	// create empty chipset
	_this.chipset_pic = new flash.display.BitmapData( 480, 256, true, 0 );
	
	// create empty collision array
	_this.collision_array = new Array();
	for(var x=0; x<width; x++)
	{
		_this.collision_array[x] = new Array();
		for(var y=0; y<height; y++)
		{
			_this.collision_array[x][y] = 0;
		}// for y
	}// for x
	#include "addCollisionMethods.as"
	addCollisionMethods( _this.collision_array );
	
	
	
	
	
	// ________________________________________________________
	// 	FUNCTIONS
	
	_this.reset = function( width, height )
	{
		// remove movieClips
		// remove images
		for (var nam in _this)
		{
			var thisObj = _this[nam];
			// remove layers
			if( typeof(thisObj) == "movieclip" )
			{
				if (nam.slice(0, 5) == "layer")
				{
					thisObj.removeMovieClip();
				}
			}
			// remove layer images  &  chipset
			else if( thisObj instanceof flash.display.BitmapData )
			{
				if (nam.slice(0, 5) == "layer"  ||
					nam == "chipset_pic")
				{
					_this[nam].dispose();
					delete _this[nam];		// delete original reference
				}
			}
		}// for-in:  _this
		
		// reset variables
		_this.width = (width) ? width : _this.width;		// if new size isn't specified, retain existing size
		_this.height = (height) ? height : _this.height;
		// reset chipset image
		_this.chipset_pic = new flash.display.BitmapData( 480, 256, true, 0 );
		//var destArea = new flash.geom.Rectangle( 0, 0, 480, 256 );
		//_this.chipset_pic.fillRect( destArea, 0 );
		
		// create empty collision array
		_this.collision_array = new Array();
		for(var x=0; x<_this.width; x++)
		{
			_this.collision_array[x] = new Array();
			for(var y=0; y<_this.height; y++)
			{
				_this.collision_array[x][y] = 0;
			}// for y
		}// for x
		#include "addCollisionMethods.as"
		addCollisionMethods( _this.collision_array );
	}// reset()
	
	
	
	
	
	_this.setTile = function(x, y, id, layer)
	{
		// resolve optional parameters
		var layer = (layer != undefined) ? layer : 0;
		
		// determine copy coordinates, based on id
		var chipWidth = 30;
		var xChip = id % chipWidth;
		var yChip = Math.floor( id / chipWidth );
		var copyArea = new flash.geom.Rectangle(
											xChip*16,
											yChip*16,
											16,
											16);
		
		// determine paste coordinates
		var pastePoint = new flash.geom.Point(
											  x*16,
											  y*16);	
		
		// create layer if neccessary
		var layerName = "layer"+layer+"_mc";			// layer2_mc
		var imageName = "layer"+layer+"_pic";		// layer2_pic
		if( _this[layerName] == undefined )
		{// if:  layer doesn't exist
			// image
			var thisImage = _this[imageName] = new flash.display.BitmapData(
															_this.width*16,
															_this.height*16,
															true,
															0);
			// movieClip
			_this.createEmptyMovieClip( layerName, layer*10 );
			_this[layerName].attachBitmap( thisImage, 0, true );
		}// if:  layer doesn't exist
		
		// copy the tile image to the layer
		_this[imageName].copyPixels( _this.chipset_pic, copyArea, pastePoint );
	}// setTile()
	
	
	
	
	
	_this.loadChipset = function( fileName )
	{
		// clear image
		var fillArea = new flash.geom.Rectangle(0,0,480,256);
		_this.chipset_pic.fillRect( fillArea, 0 );
		// load image
		_this.createEmptyMovieClip( "load_mc", 99 );
		_this.loader = new MovieClipLoader();
		_this.loader.loadClip( fileName, _this.load_mc );
		_this.loader.onLoadInit = function( load_mc )
		{
			// copy loaded image
			// // snapshot
			var newChip_pic = new flash.display.BitmapData( 480, 256, true, 0 );
			newChip_pic.draw( _this.load_mc );
			// // copy
			var copyArea = new flash.geom.Rectangle( 0, 0, 480, 256 );
			_this.chipset_pic.copyPixels( newChip_pic, copyArea, {x:0,y:0} );
			
			// call external function
			_this.onLoad();		// externally-defined function
			
			// // clean-up
			_this.loader.unloadClip( load_mc );
			delete _this.loader;
		}// chipset loaded()
	}// loadChipset()
	
	
	
	
	
	_this.drawObj = function( data_obj )
	{
		// reset
		_this.reset();
		
		// width
		_this.width = data_obj.width;
		
		// height
		_this.height = data_obj.height;
		
		// collision
		if( data_obj.collision != undefined)
		{// if:  collision data exists
			_this.collision_array = new Array();
			for(var x=0; x<data_obj.width; x++)
			{
				_this.collision_array[x] = new Array();
				for(var y=0; y<data_obj.height; y++)
				{
					_this.collision_array[x][y] = data_obj.collision[x][y];
				}// for y
			}// for x
			#include "addCollisionMethods.as"
			addCollisionMethods( _this.collision_array );
		}// if:  collision data exists
		
		
		
		// chipset_pic
		// // clear image
		var fillArea = new flash.geom.Rectangle(0,0,480,256);
		_this.chipset_pic.fillRect( fillArea, 0 );
		// // load image
		_this.createEmptyMovieClip( "load_mc", 99 );
		_this.loader = new MovieClipLoader();
		_this.loader.loadClip( data_obj.chipset, _this.load_mc );
		_this.loader.onLoadInit = function( load_mc )
		{
			// // copy loaded image
			// // // snapshot
			var newChip_pic = new flash.display.BitmapData( 480, 256, true, 0 );
			newChip_pic.draw( _this.load_mc );
			// // // copy
			var copyArea = new flash.geom.Rectangle( 0, 0, 480, 256 );
			_this.chipset_pic.copyPixels( newChip_pic, copyArea, {x:0,y:0} );
			// // // clean-up
			_this.loader.unloadClip( _this.load_mc );
			
			
			
			// draw layers  (setTile)
			for (lay=0; lay<data_obj.layers.length; lay++)
			{
				for (var x=0; x<_this.width; x++)
				{
					for (var y=0; y<_this.height; y++)
					{
						var id = data_obj.layers[lay][x][y];
						_this.setTile( x, y, id, lay );
					}// for:  width
				}// for:  height
			}// for:  layers
			
			
			// call external function
			_this.onLoad();		// externally-defined function
			
			
			// clean-up
			delete _this.loadData;		// In case load() called this function, then clean up after load() as well.
			delete _this.loader;
		}// chipset loaded()
	}// drawObj()
	
	
	
	
	
	_this.load = function( fileName )
	{
		// load XML
		XML.prototype.ignoreWhite = true;
		_this.load_xml = new XML();
		_this.load_xml.load( fileName );
		_this.load_xml.onLoad = function( success )
		{
			#include "readXml.as"
			if( success )
			{
				
				// convert XML
				_this.loadData = new Object();
				readXml( _this.load_xml, _this.loadData );
				
				// pass object to drawObj()
				_this.drawObj( _this.loadData.map );
				
			}// if:  success
		}// onLoad()
	}// load()
	
	
	
	_this.scroll = function( x, y, screenWidth, screenHeight)
	{
		// resolve optional parameters
		var screenWidth = (screenWidth) ? screenWidth : 320;
		var screenHeight = (screenHeight) ? screenHeight : 240;
		// calculate view screen center
		var halfWidth = screenWidth / 2;
		var halfHeight = screenHeight / 2;
		// get the map's pixel width
		var mapWidth = _this.width * 16;
		var mapHeight = _this.height * 16;
		
		
		// horz
		if( x < halfWidth)
		{
			// far-left
			_this._x = 0;
		}
		else if( x > mapWidth-halfWidth)
		{
			// far-right
			_this._x = -mapWidth+screenWidth;
		}
		else
		{
			// scroll
			_this._x = -x + halfWidth;
		}
		
		
		// vert
		if( y < halfHeight)
		{
			// top
			_this._y = 0;
		}
		else if( y > mapHeight-halfHeight)
		{
			// bottom
			_this._y = -mapHeight+screenHeight;
		}
		else
		{
			// scroll
			_this._y = -y + halfHeight;
		}
	}// scroll()
	
	
	
	// ________________________________________________________
	
	
	
	// return reference to this map
	return _this;
}// makeMap()